//
//  GyLoginListController.m
//  GyDemo
//
//  Created by ak on 2021/2/23.
//  Copyright © 2021 getui. All rights reserved.
//

#import "LoginListController.h"
#import "LoginListCell.h"
#import "SuccessController.h"
#import "WechatView.h"
#import "CustomLoginView.h"


typedef NS_ENUM(NSInteger, Mode) {
    ModeFullScreen = 1,
    ModePop,
    ModeWindow,
    ModeLandscape
};
@interface LoginListController ()<UITableViewDataSource, UITableViewDelegate>
@property(nonatomic, strong) UIView *backgroundView;
@property(nonatomic, strong) CAGradientLayer *gradientLayer;
@property(nonatomic, strong) UITableView *tableView;
@property(nonatomic, strong) NSArray *items;
@property(nonatomic, assign) CFAbsoluteTime startTime;
@property(nonatomic, strong) WechatView *chatView;
@property(nonatomic, strong) CustomLoginView *customView;
@property(nonatomic, strong) UIView *contentView;


@property(nonatomic, assign) Mode mode;
@end

@implementation LoginListController

- (void)viewWillAppear:(BOOL)animated {
    [super viewWillAppear:animated];
    dispatch_after(dispatch_time(DISPATCH_TIME_NOW, NSEC_PER_SEC * 1), dispatch_get_main_queue(), ^{
        [Utils switchNewOrientation:UIInterfaceOrientationMaskPortrait];
    });
}

- (void)viewDidLoad {
    [super viewDidLoad];
    self.navigationController.navigationBar.tintColor = UIColor.whiteColor;
    [self setupView];
}

//MARK: - Action

- (void)fullScreenAction {
    self.mode = ModeFullScreen;
    [self.contentView removeFromSuperview];
    self.contentView.frame = self.view.bounds;
    [self.customView layoutFullScreen];
    GyAuthViewModel *model = [self customModel];
    model.pullAuthVCStyle = OLPullAuthVCStylePush;   //push
    //model.pullAuthVCStyle = OLPullAuthVCStyleModal;
    [self login:model];
}

- (void)floatingWindowAction {
    self.mode = ModeWindow;
    [self.contentView removeFromSuperview];
    self.contentView.frame = self.view.bounds;
    [self.customView layoutFloatingWindow];
    GyAuthViewModel *model = [self customModel];
    model.pullAuthVCStyle = OLPullAuthVCStyleModal; //present
    model.pullAnimate = NO;
    [self login:model];
}

- (void)popupAction {
    self.mode = ModePop;
    [self.contentView removeFromSuperview];
    self.contentView.frame = self.view.bounds;
    [self.customView layoutPopup];
    GyAuthViewModel *model = [self customModel];
    model.pullAuthVCStyle = OLPullAuthVCStyleModal;
    [self login:model];
}

- (void)landscapeAction {
    self.mode = ModeLandscape;
    [self.contentView removeFromSuperview];
    self.contentView.frame = CGRectMake(0, 0, kScreenHeight, kScreenWidth);
    [self.customView layoutLandscape];
    GyAuthViewModel *model = [self customModel];
    model.pullAuthVCStyle = OLPullAuthVCStylePush;
    [self login:model];
}

- (UIView *)contentView {
    if (!_contentView) {
        _contentView = [[UIView alloc] initWithFrame:self.view.bounds];
        _contentView.backgroundColor = [UIColor clearColor];
        [_contentView addSubview:self.customView];
    }
    return _contentView;
}

- (CustomLoginView *)customView {
    if (!_customView) {
        _customView = (CustomLoginView*) [[[UINib nibWithNibName:@"CustomLoginView" bundle:nil] instantiateWithOwner:nil options:nil] firstObject];
        __weak LoginListController *weakSelf = self;
        _customView.jumpWebClick = ^(NSURL *url) {
            WebViewController *webVC = [WebViewController new];
            webVC.url = url;
            UINavigationController *nav = [[UINavigationController alloc] initWithRootViewController:webVC];
            if (weakSelf.mode == ModeFullScreen) {
                [weakSelf.navigationController pushViewController:webVC animated:YES];
            } else {
                
                UIImage *img = [UIImage imageNamed:@"TYRZResource.bundle/back_13x21_.png"];
                UIButton*btn = [[UIButton alloc] initWithFrame:CGRectZero];
                btn.frame = CGRectMake(0, 0, 40, 40);
                [btn setImage:img forState:UIControlStateNormal];
                [btn addTarget:webVC action:@selector(back) forControlEvents:UIControlEventTouchUpInside];
                webVC.navigationItem.leftBarButtonItem = [[UIBarButtonItem alloc] initWithCustomView:btn];
                //因为当前登录页是present方式，所以网页详情也用present
                [[GeYanSdk currentAuthViewController] presentViewController:nav animated:YES completion:nil];
            }
            
        };
    }
    return _customView;
}

- (void)back {
    [GeYanSdk closeAuthVC:nil];
}

//MARK: - Model
- (GyAuthViewModel *)customModel {
    GyAuthViewModel *viewModel = [GyAuthViewModel new];
    viewModel.statusBarStyle = UIStatusBarStyleDefault;
    //暗黑模式
    viewModel.userInterfaceStyle = @(0);
    
    __weak LoginListController *weakSelf = self;
    
    viewModel.viewLifeCycleBlock = ^(NSString *viewLifeCycle, BOOL animated) {
        NSLog(@"页面生命周期 回调: %@, %@", viewLifeCycle, @(animated));
        UIViewController *controller = [GeYanSdk currentAuthViewController];
        if ([viewLifeCycle isEqualToString:@"viewDidLoad"]) {
            controller.view.backgroundColor = [UIColor clearColor];
            if (weakSelf.mode == ModeFullScreen || weakSelf.mode == ModeLandscape) {
                //添加导航,返回箭头按钮
                UIImage *img = [UIImage imageNamed:@"TYRZResource.bundle/back_13x21_.png"];
                UIButton*btn = [[UIButton alloc] initWithFrame:CGRectZero];
                btn.frame = CGRectMake(0, 0, 40, 40);
                [btn setImage:img forState:UIControlStateNormal];
                [btn addTarget:weakSelf action:@selector(back) forControlEvents:UIControlEventTouchUpInside];
                controller.navigationItem.leftBarButtonItem = [[UIBarButtonItem alloc] initWithCustomView:btn];
            }
        }
        
        if ([viewLifeCycle isEqualToString:@"viewDidAppear"]) {
            //浮窗模式的动画
            if (weakSelf.mode == ModeWindow) {
                [UIView animateWithDuration:0.25 animations:^{
                    CGRect frame = weakSelf.customView.frame;
                    frame.origin.y = kScreenHeight - 540;
                    weakSelf.customView.frame = frame;
                }];
            }
        }
    };
    
    viewModel.clickAuthButtonBlock = ^() {
        weakSelf.startTime = NSDate.new.timeIntervalSince1970;
        NSLog(@"点击登录按钮 回调");
        [CommonToastHUD showLoadingHUDWithMessage:@"登录中"];
        //返回YES:流程继续， 返回NO:登录中断
        return YES;
    };
    return viewModel;
}

//MARK: - Login

- (void)login:(GyAuthViewModel *)viewModel {
    __weak LoginListController *weakSelf = self;
    NSArray *list = @[self.customView.pnLabel,
                      self.customView.sloganLabel,
                      self.customView.loginBtn,
                      self.customView.checkboxBtn,
                      self.customView.termLabel];
    // 一键登录
    /*
     由SDK控制Controller生命周期，开发者控制Controller中View的显示。
     但View的显示需要遵循一定规则：
     1 pnLabel      手机号码
     2 sloganLabel  运营商slogan
     3 loginBtn     登录按钮
     4 checkboxBtn  协议勾选框
     5 termLabel    协议内容
     以上控件必须保持页面内可见，不可遮挡、隐藏、曲解意义等操作。
     以上控件的内容值，在预取号成功后可以通过接口[GyContentModel currentModel]获取。
     */
    if ([GeYanSdk isPreGettedTokenValidate]) {
        //渲染页面信息
        [self.customView render];
        // 一键登录预取号还在有效期，可直接调用oneTapLogin方法进行登录
        [Utils switchNewOrientation:viewModel.supportedInterfaceOrientations];
        [GeYanSdk oneTapLogin:self
                    viewModel:viewModel
                  contentView:self.contentView
                  subViewList:list
                   loginBlock:^(NSDictionary * _Nullable dic) {
            //登录完成
            [weakSelf loginFinish:dic];
        }];
        return;
    }
    
    [GeYanSdk preGetToken:^(NSDictionary *preDic) {
        NSLog(@"preGetToken: %@ %@", preDic, preDic[@"msg"]);
        [CommonToastHUD hideAllHUD];
        if (![GeYanSdk isPreGettedTokenValidate]) {
            [CommonToastHUD hideAllAndTip:[Utils dic2Str:preDic]];
            return;
        }
        [Utils switchNewOrientation:viewModel.supportedInterfaceOrientations];
        //渲染页面信息
        [weakSelf.customView render];
        [GeYanSdk oneTapLogin:weakSelf
                    viewModel:viewModel
                  contentView:weakSelf.contentView
                  subViewList:list
                   loginBlock:^(NSDictionary * _Nullable dic) {
            //登录完成
            [weakSelf loginFinish:dic];
        }];
    }];
}

- (void)loginFinish:(NSDictionary *)result {
    NSLog(@"oneTapLogin:%@ %@ %@", result, result[@"msg"], result[@"metadata"][@"msg"] ?: @"");
    [CommonToastHUD hideAllHUD];
    if (![result[@"code"] isEqualToNumber:@30000]) {
        [CommonToastHUD hideAllAndTip:[Utils dic2Str:result]];
        return;
    }
    [self queryPhoneNumber:result];
    [GeYanSdk closeAuthVC:YES completion:^{
        NSLog(@"关闭授权页面");
    }];
}

/// 向服务端查询完整手机号
- (void)queryPhoneNumber:(NSDictionary*)result {
    [CommonToastHUD showLoadingHUDWithMessage:@"查询号码中"];
    // TODO, 正式版本请根据服务端对接文档使用加密接口 https://openapi-gy.getui.com/v2/gy/ct_login/gy_get_pn
    NSURL *url = [[NSURL alloc] initWithString:@"https://openapi-gy.getui.com/v1/gy/ct_login/gy_get_pn"];
    NSString *token = result[@"token"] ?:@"";
    NSString *pid = result[@"processID"] ?:@"";
    NSString *gyuid = result[@"gyuid"] ?:@"";
    AppDelegate *appDelegate = (AppDelegate *)[[UIApplication sharedApplication] delegate];
    NSDictionary *params = @{
        @"appId": [appDelegate getAppID],
        @"gyuid": gyuid,
        @"token": token,
        @"processId": pid
    };
    NSLog(@"json : %@", params);
    __weak LoginListController *weakSelf = self;
    NSData *data = [Utils dic2JsonData:params];
    NSMutableURLRequest *request = [NSMutableURLRequest requestWithURL:url];
    [request setHTTPMethod:@"POST"];
    [request setValue:@"application/json" forHTTPHeaderField:@"Content-Type"];
    [request setHTTPBody:data];
    self.startTime = CFAbsoluteTimeGetCurrent();
    NSURLSession *session = [NSURLSession sessionWithConfiguration:[NSURLSessionConfiguration defaultSessionConfiguration] delegate:nil delegateQueue:nil];
    NSURLSessionDataTask *task = [session dataTaskWithRequest:request completionHandler:^(NSData *_Nullable data, NSURLResponse *_Nullable response, NSError *_Nullable error) {
        dispatch_async(dispatch_get_main_queue(), ^{
            [CommonToastHUD hideAllHUD];
            if (!data) {
                [CommonToastHUD showToastWithMessage:@"查询号码失败，请重试"];
                return;
            }
            CFAbsoluteTime endTime = CFAbsoluteTimeGetCurrent();
            //{"errno":0,"data":{"msg":"成功","result":"20000","data":{"pn":"15258060533"}}}
            NSDictionary *dic = [Utils jsonData2Dic:data];
            NSLog(@"收到回执: %lf", endTime);
            NSLog(@"收到回执内容：%@", dic);
            NSLog(@"取号耗时: %lf，token：%@", (endTime - weakSelf.startTime) * 1000, token);
            
            if ([dic[@"errno"] intValue] == 0) {
                SuccessController *ctrl = [[SuccessController alloc] init];
                ctrl.tipTitle = @"登录成功";
                ctrl.pn = dic[@"data"][@"data"][@"pn"]?:@"";
                [weakSelf.navigationController pushViewController:ctrl animated:YES];
            } else {
                [CommonToastHUD hideAllAndTip:@"查询失败"];
            }
        });
    }];
    [task resume];
}

//MARK: - 自定义动画

- (void)addPresentAnimation {
    // 自定义弹窗动画
    /*
     pageCurl            向上翻一页
     pageUnCurl          向下翻一页
     rippleEffect        滴水效果
     suckEffect          收缩效果，如一块布被抽走
     cube                立方体效果
     oglFlip             上下翻转效果
     */
    CATransition *animation = [CATransition animation];
    animation.duration = 1.25;
    animation.timingFunction = [CAMediaTimingFunction functionWithName:kCAMediaTimingFunctionEaseInEaseOut];
    animation.type = @"oglFlip";
    animation.subtype = kCATransitionFromLeft;
    UIViewController *vc = [GeYanSdk currentAuthViewController];
    vc.modalPresentationStyle = UIModalPresentationOverFullScreen;
    [self.view.window.layer addAnimation:animation forKey:nil];
}

- (void)addDismissAnimation {
    CATransition *animation = [CATransition animation];
    animation.duration = 1.25;
    animation.timingFunction = [CAMediaTimingFunction functionWithName:kCAMediaTimingFunctionEaseInEaseOut];
    animation.type = @"oglFlip";
    animation.subtype = kCATransitionFromLeft;
    UIViewController *vc = [GeYanSdk currentAuthViewController];
    [vc.view.window.layer addAnimation:animation forKey:nil];
}

//MARK: - TableView Deleagte

- (NSInteger)tableView:(UITableView *)tableView numberOfRowsInSection:(NSInteger)section {
    return self.items.count;
}

- (UITableViewCell *)tableView:(UITableView *)tableView cellForRowAtIndexPath:(NSIndexPath *)indexPath {
    LoginListCell *cell = [tableView dequeueReusableCellWithIdentifier:@"LoginListCell" forIndexPath:indexPath];
    cell.item = self.items[indexPath.row];
    return cell;
}

- (void)tableView:(UITableView *)tableView didSelectRowAtIndexPath:(NSIndexPath *)indexPath {
    switch (indexPath.row) {
        case 0:
            [GeYanSdk preGetToken:^(NSDictionary * _Nullable preDic) {
                [CommonToastHUD hideAllAndTip:[Utils dic2Str:preDic]];
                NSLog(@"preGetToken: %@ %@", preDic, preDic[@"msg"]);
            }];
            break;
        case 1:
            [self fullScreenAction];
            break;
        case 2:
            [self floatingWindowAction];
            break;
        case 3:
            [self popupAction];
            break;
        case 4:
            [self landscapeAction];
            break;
        default:
            break;
    }
}

//MARK: - Private

- (void)setupView {
    self.navigationItem.title = @"授权页样式选择";
    self.view.backgroundColor = [UIColor colorWithRed:39/255.0 green:132/255.0 blue:241/255.0 alpha:1];
    __weak LoginListController *weakSelf = self;
    [self.view addSubview:self.backgroundView];
    [self.backgroundView mas_makeConstraints:^(MASConstraintMaker *make) {
        make.edges.equalTo(weakSelf.view);
    }];
    UILabel *subtitleLabel = [[UILabel alloc] init];
    subtitleLabel.text = @"不同场景选择不同的APP授权页样式";
    subtitleLabel.textColor = UIColor.whiteColor;
    subtitleLabel.font = [UIFont boldSystemFontOfSize:16];
    subtitleLabel.textAlignment = NSTextAlignmentCenter;
    [self.backgroundView addSubview:subtitleLabel];
    [subtitleLabel mas_makeConstraints:^(MASConstraintMaker *make) {
        make.left.right.mas_equalTo(weakSelf.backgroundView);
        make.height.mas_equalTo(35);
        make.top.mas_equalTo(kNavigationHeight);
    }];
    [self.backgroundView addSubview:self.tableView];
    [self.tableView mas_makeConstraints:^(MASConstraintMaker *make) {
        make.top.equalTo(subtitleLabel.mas_bottom);
        make.left.right.bottom.mas_equalTo(weakSelf.backgroundView);
    }];
}

- (void)viewDidLayoutSubviews {
    [super viewDidLayoutSubviews];
    self.gradientLayer.frame = UIScreen.mainScreen.bounds;
}

- (UIView *)backgroundView {
    if (!_backgroundView) {
        _backgroundView = [[UIView alloc] initWithFrame:self.view.bounds];
        CAGradientLayer *gradient = [CAGradientLayer layer];
        gradient.frame = self.view.bounds;
        gradient.colors = @[(id)[UIColor colorWithRed:39/255.0 green:132/255.0 blue:241/255.0 alpha:1].CGColor,
                            (id)[UIColor colorWithRed:30/255.0 green:125/255.0 blue:235/255.0 alpha:1].CGColor,
                            (id)[UIColor colorWithRed:128/255.0 green:202/255.0 blue:255/255.0 alpha:1].CGColor];
        gradient.startPoint = CGPointMake(0, 0);
        gradient.endPoint = CGPointMake(1, 1);
        self.gradientLayer = gradient;
        [_backgroundView.layer addSublayer:gradient];
    }
    return _backgroundView;
}

- (UITableView *)tableView {
    if (!_tableView) {
        _tableView = [[UITableView alloc] initWithFrame:CGRectZero];
        _tableView.backgroundColor = UIColor.clearColor;
        _tableView.tableFooterView = [UIView new];
        [_tableView registerClass:[LoginListCell class] forCellReuseIdentifier:@"LoginListCell"];
        _tableView.dataSource = self;
        _tableView.delegate = self;
        _tableView.rowHeight = 165*kScreenScale;
        _tableView.separatorStyle = UITableViewCellSeparatorStyleNone;
    }
    return _tableView;
}

- (NSArray *)items {
    if (!_items) {
        _items = @[@[@"login_full",@"预取号",@"预取号"],
                   @[@"login_full",@"全屏式",@"页面简洁沉浸视觉体验"],
                   @[@"login_float",@"浮窗式",@"方便操作不影响别的窗口"],
                   @[@"login_alert",@"弹窗式",@"页面中心位置吸引用户注意"],
                   @[@"login_landscape",@"横屏式",@"打游戏看电视的首选模式"]];
    }
    return _items;
}

- (void)dealloc {
    NSLog(@"dealloc %@", self);
}

@end
